// Yol Yardım Poliçesi JavaScript Fonksiyonları

$(document).ready(function() {
    
    // Policy card selection
    $('input[name="policy_type"]').change(function() {
        $('.policy-card').removeClass('selected');
        $(this).closest('.policy-card').addClass('selected');
        
        // Animate selection
        $(this).closest('.policy-card').addClass('animate__animated animate__pulse');
        setTimeout(() => {
            $(this).closest('.policy-card').removeClass('animate__animated animate__pulse');
        }, 600);
    });

    // Form validation
    $('.contact-form-validated').validate({
        rules: {
            first_name: {
                required: true,
                minlength: 2,
                maxlength: 50
            },
            last_name: {
                required: true,
                minlength: 2,
                maxlength: 50
            },
            phone: {
                required: true,
                minlength: 10,
                maxlength: 15,
                phoneNumber: true
            },
            email: {
                required: true,
                email: true,
                maxlength: 100
            },
            tc_no: {
                required: true,
                minlength: 11,
                maxlength: 11,
                digits: true,
                tcNumber: true
            },
            birth_date: {
                required: true,
                dateISO: true,
                minAge: true
            },
            license_plate: {
                required: true,
                plateNumber: true
            },
            vehicle_brand: "required",
            vehicle_model: {
                required: true,
                minlength: 2,
                maxlength: 50
            },
            vehicle_year: {
                required: true,
                min: 1990,
                max: new Date().getFullYear() + 1
            },
            policy_type: "required",
            terms_accepted: "required"
        },
        messages: {
            first_name: {
                required: "Lütfen adınızı girin",
                minlength: "Ad en az 2 karakter olmalıdır",
                maxlength: "Ad en fazla 50 karakter olmalıdır"
            },
            last_name: {
                required: "Lütfen soyadınızı girin",
                minlength: "Soyad en az 2 karakter olmalıdır",
                maxlength: "Soyad en fazla 50 karakter olmalıdır"
            },
            phone: {
                required: "Lütfen telefon numaranızı girin",
                minlength: "Telefon numarası en az 10 haneli olmalıdır",
                maxlength: "Telefon numarası en fazla 15 haneli olmalıdır",
                phoneNumber: "Geçerli bir telefon numarası girin"
            },
            email: {
                required: "Lütfen e-mail adresinizi girin",
                email: "Lütfen geçerli bir e-mail adresi girin",
                maxlength: "E-mail adresi en fazla 100 karakter olmalıdır"
            },
            tc_no: {
                required: "Lütfen TC kimlik numaranızı girin",
                minlength: "TC kimlik numarası 11 haneli olmalıdır",
                maxlength: "TC kimlik numarası 11 haneli olmalıdır",
                digits: "TC kimlik numarası sadece rakam içermelidir",
                tcNumber: "Geçersiz TC kimlik numarası"
            },
            birth_date: {
                required: "Lütfen doğum tarihinizi seçin",
                dateISO: "Geçerli bir tarih girin",
                minAge: "18 yaşından küçük olamazsınız"
            },
            license_plate: {
                required: "Lütfen araç plakasını girin",
                plateNumber: "Geçerli bir plaka formatı girin (örn: 34ABC123)"
            },
            vehicle_brand: "Lütfen araç markasını seçin",
            vehicle_model: {
                required: "Lütfen araç modelini girin",
                minlength: "Model en az 2 karakter olmalıdır",
                maxlength: "Model en fazla 50 karakter olmalıdır"
            },
            vehicle_year: {
                required: "Lütfen model yılını girin",
                min: "Model yılı 1990'dan büyük olmalıdır",
                max: "Geçersiz model yılı"
            },
            policy_type: "Lütfen bir poliçe paketi seçin",
            terms_accepted: "Lütfen kullanım şartlarını kabul edin"
        },
        errorElement: 'span',
        errorClass: 'error',
        validClass: 'valid',
        errorPlacement: function(error, element) {
            error.insertAfter(element);
        },
        highlight: function(element) {
            $(element).addClass('error').removeClass('valid');
        },
        unhighlight: function(element) {
            $(element).removeClass('error').addClass('valid');
        },
        submitHandler: function(form) {
            submitForm(form);
        }
    });

    // Custom validation methods
    $.validator.addMethod("phoneNumber", function(value, element) {
        return this.optional(element) || /^[\+]?[\d\s\-\(\)]{10,15}$/.test(value);
    });

    $.validator.addMethod("tcNumber", function(value, element) {
        if (this.optional(element)) return true;
        
        if (value.length !== 11) return false;
        
        var digits = value.split('').map(Number);
        var sum1 = 0, sum2 = 0;
        
        for (var i = 0; i < 9; i++) {
            if (i % 2 === 0) sum1 += digits[i];
            else sum2 += digits[i];
        }
        
        var check1 = (sum1 * 7 - sum2) % 10;
        var check2 = (sum1 + sum2 + digits[9]) % 10;
        
        return check1 === digits[9] && check2 === digits[10];
    });

    $.validator.addMethod("plateNumber", function(value, element) {
        return this.optional(element) || /^(0[1-9]|[1-7][0-9]|8[01])[A-Z]{1,3}[0-9]{2,4}$/.test(value.toUpperCase());
    });

    $.validator.addMethod("minAge", function(value, element) {
        if (this.optional(element)) return true;
        
        var birthDate = new Date(value);
        var today = new Date();
        var age = today.getFullYear() - birthDate.getFullYear();
        var monthDiff = today.getMonth() - birthDate.getMonth();
        
        if (monthDiff < 0 || (monthDiff === 0 && today.getDate() < birthDate.getDate())) {
            age--;
        }
        
        return age >= 18;
    });

    // Input formatters
    $('input[name="phone"]').on('input', function() {
        var value = $(this).val().replace(/\D/g, '');
        var formatted = value.replace(/(\d{3})(\d{3})(\d{4})/, '$1 $2 $3');
        $(this).val(formatted);
    });

    $('input[name="tc_no"]').on('input', function() {
        var value = $(this).val().replace(/\D/g, '');
        $(this).val(value);
    });

    $('input[name="license_plate"]').on('input', function() {
        $(this).val($(this).val().toUpperCase());
    });

    // Real-time validation feedback
    $('.contact__form__control input, .contact__form__control select').on('blur', function() {
        $(this).valid();
    });

    // Policy card hover effects
    $('.policy-card').hover(
        function() {
            $(this).addClass('animate__animated animate__heartBeat');
        },
        function() {
            $(this).removeClass('animate__animated animate__heartBeat');
        }
    );

    // Smooth scrolling for form sections
    $('.form-section-title').on('click', function() {
        var target = $(this).next('.row, .policy-options');
        if (target.length) {
            $('html, body').animate({
                scrollTop: target.offset().top - 100
            }, 500);
        }
    });

    // Auto-save form data to localStorage
    var formData = {};
    $('.contact__form input, .contact__form select').on('change input', function() {
        formData[$(this).attr('name')] = $(this).val();
        localStorage.setItem('yolyardim_form_data', JSON.stringify(formData));
    });

    // Load saved form data
    var savedData = localStorage.getItem('yolyardim_form_data');
    if (savedData) {
        try {
            var data = JSON.parse(savedData);
            Object.keys(data).forEach(function(key) {
                var element = $('[name="' + key + '"]');
                if (element.length) {
                    if (element.is('input[type="radio"]')) {
                        element.filter('[value="' + data[key] + '"]').prop('checked', true).trigger('change');
                    } else if (element.is('input[type="checkbox"]')) {
                        element.prop('checked', data[key] === 'on');
                    } else {
                        element.val(data[key]);
                    }
                }
            });
        } catch (e) {
            console.log('Saved form data corrupted, clearing...');
            localStorage.removeItem('yolyardim_form_data');
        }
    }
});

// Form submission handler
function submitForm(form) {
    var $form = $(form);
    var $submitBtn = $form.find('button[type="submit"]');
    var originalText = $submitBtn.find('span').first().text();
    
    // Disable form and show loading
    $form.addClass('form-loading');
    $submitBtn.prop('disabled', true);
    $submitBtn.find('span').first().text('İşleniyor...');
    
    // Collect form data
    var formData = new FormData(form);
    
    // AJAX submission
    $.ajax({
        url: 'process_policy.php',
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function(response) {
            try {
                var result = JSON.parse(response);
                if (result.success) {
                    showAlert('Poliçe başvurunuz başarıyla alındı! En kısa sürede sizinle iletişime geçeceğiz.', 'success');
                    form.reset();
                    $('.policy-card').removeClass('selected');
                    localStorage.removeItem('yolyardim_form_data');
                    
                    // Redirect to success page or payment
                    if (result.redirect) {
                        setTimeout(function() {
                            window.location.href = result.redirect;
                        }, 2000);
                    }
                } else {
                    showAlert(result.message || 'Bir hata oluştu. Lütfen tekrar deneyin.', 'error');
                }
            } catch (e) {
                showAlert('Sunucu yanıtı işlenirken hata oluştu.', 'error');
            }
        },
        error: function(xhr, status, error) {
            showAlert('Bağlantı hatası. Lütfen internet bağlantınızı kontrol edin.', 'error');
        },
        complete: function() {
            // Re-enable form
            $form.removeClass('form-loading');
            $submitBtn.prop('disabled', false);
            $submitBtn.find('span').first().text(originalText);
        }
    });
}

// Alert system
function showAlert(message, type) {
    var alertClass = 'alert-' + type;
    var alertHtml = '<div class="alert ' + alertClass + '">' + message + '</div>';
    
    // Remove existing alerts
    $('.alert').remove();
    
    // Add new alert
    $('.contact-two__content').prepend(alertHtml);
    
    // Scroll to alert
    $('html, body').animate({
        scrollTop: $('.alert').offset().top - 100
    }, 500);
    
    // Auto remove after 5 seconds
    setTimeout(function() {
        $('.alert').fadeOut(500, function() {
            $(this).remove();
        });
    }, 5000);
}

// Price calculator (optional feature)
function calculatePrice() {
    var policyType = $('input[name="policy_type"]:checked').val();
    var vehicleYear = parseInt($('input[name="vehicle_year"]').val());
    var basePrice = 0;
    
    switch (policyType) {
        case 'basic':
            basePrice = 299;
            break;
        case 'premium':
            basePrice = 499;
            break;
        case 'platinum':
            basePrice = 799;
            break;
    }
    
    // Vehicle age discount/surcharge
    var currentYear = new Date().getFullYear();
    var vehicleAge = currentYear - vehicleYear;
    
    if (vehicleAge > 15) {
        basePrice += 50; // Old vehicle surcharge
    } else if (vehicleAge < 3) {
        basePrice -= 30; // New vehicle discount
    }
    
    return basePrice;
}

// Export functions for external use
window.YolYardim = {
    showAlert: showAlert,
    calculatePrice: calculatePrice,
    submitForm: submitForm
};