// YurtAssist - Ortak JavaScript Dosyası - assets/js/script.js

// Global değişkenler
let selectedPackage = null;

// Sayfa yüklendiğinde
document.addEventListener('DOMContentLoaded', function() {
    initializePage();
});

// Sayfa başlatma
function initializePage() {
    // Form validasyonları
    setupFormValidations();
    
    // Sayfa bazlı işlemler
    const currentPage = getCurrentPage();
    
    switch(currentPage) {
        case 'index':
            initializePackageSelection();
            break;
        case 'customer-info':
            initializeCustomerForm();
            break;
        case 'payment':
            initializePaymentForm();
            break;
        case 'customer-login':
            initializeLoginForm();
            break;
    }
}

// Mevcut sayfayı tespit et
function getCurrentPage() {
    const path = window.location.pathname;
    const filename = path.split('/').pop().split('.')[0];
    return filename || 'index';
}

// Paket seçimi fonksiyonları
function selectPackage(packageId) {
    console.log('Paket seçiliyor:', packageId);
    
    const packageCard = document.querySelector(`[data-package-id="${packageId}"]`);
    if (!packageCard) {
        console.error('Paket kartı bulunamadı:', packageId);
        return;
    }
    
    const packageName = packageCard.querySelector('h3').textContent;
    const packagePrice = packageCard.dataset.price;
    
    selectedPackage = {
        id: packageId,
        name: packageName,
        price: packagePrice,
        details: getPackageDetails(packageCard)
    };
    
    console.log('Seçilen paket:', selectedPackage);
    
    // Diğer paketleri gizle
    hideOtherPackages(packageId);
    
    // Seçili paket bilgisini göster
    showSelectedPackage();
}

function getPackageDetails(packageCard) {
    const details = {};
    const detailItems = packageCard.querySelectorAll('.detail-item');
    
    detailItems.forEach(item => {
        const label = item.querySelector('.detail-label').textContent;
        const value = item.querySelector('.detail-value').textContent;
        details[label] = value;
    });
    
    return details;
}

function hideOtherPackages(selectedId) {
    const allPackages = document.querySelectorAll('.package-card');
    
    allPackages.forEach(package => {
        if (package.dataset.packageId !== selectedId.toString()) {
            // Smooth hide animation
            package.style.transition = 'all 0.5s ease';
            package.style.opacity = '0';
            package.style.transform = 'scale(0.8)';
            
            setTimeout(() => {
                package.style.display = 'none';
            }, 500);
        } else {
            // Highlight selected package
            package.style.transform = 'scale(1.02)';
            package.style.boxShadow = '0 8px 25px rgba(30, 64, 175, 0.3)';
            package.style.border = '2px solid var(--primary-blue)';
        }
    });
}

function showSelectedPackage() {
    const selectedInfo = document.getElementById('selectedPackageInfo');
    const selectedContent = document.getElementById('selectedContent');
    
    if (selectedPackage) {
        selectedContent.innerHTML = `
            <div class="package-summary">
                <h4>${selectedPackage.name}</h4>
                <div class="summary-details">
                    ${Object.entries(selectedPackage.details).map(([key, value]) => 
                        `<div class="summary-item">
                            <span class="summary-label">${key}</span>
                            <span class="summary-value">${value}</span>
                        </div>`
                    ).join('')}
                </div>
                <div class="summary-price">
                    <strong>Toplam: ${selectedPackage.price}.00 TL (KDV Dahil: ${(parseFloat(selectedPackage.price) * 1.18).toFixed(2)} TL)</strong>
                </div>
            </div>
        `;
        
        selectedInfo.style.display = 'block';
        
        // Smooth scroll to selected package
        selectedInfo.scrollIntoView({ 
            behavior: 'smooth',
            block: 'center'
        });
        
        console.log('Seçili paket gösteriliyor:', selectedPackage);
    }
}

function goBack() {
    const allPackages = document.querySelectorAll('.package-card');
    const selectedInfo = document.getElementById('selectedPackageInfo');
    
    // Tüm paketleri tekrar göster
    allPackages.forEach(package => {
        package.style.display = 'block';
        package.style.opacity = '1';
        package.style.transform = 'scale(1)';
    });
    
    // Seçili paket bilgisini gizle
    selectedInfo.style.display = 'none';
    
    // Seçimi temizle
    selectedPackage = null;
    
    // Sayfa başına scroll yap
    window.scrollTo({ 
        top: 0, 
        behavior: 'smooth' 
    });
    
    console.log('Paket seçimi iptal edildi');
}

function goToCustomerInfo() {
    if (selectedPackage) {
        // Seçili paketi localStorage'a kaydet (sessionStorage yerine)
        localStorage.setItem('selectedPackage', JSON.stringify(selectedPackage));
        // Müşteri bilgileri sayfasına yönlendir
        window.location.href = 'customer-info.php';
    } else {
        showMessage('Lütfen bir paket seçin.', 'error');
    }
}

// Paket seçimi başlatma
function initializePackageSelection() {
    // localStorage'dan seçili paketi kontrol et
    const savedPackage = localStorage.getItem('selectedPackage');
    if (savedPackage) {
        selectedPackage = JSON.parse(savedPackage);
    }
}

// Müşteri formu işlemleri
function initializeCustomerForm() {
    const savedPackage = localStorage.getItem('selectedPackage');
    if (!savedPackage) {
        window.location.href = 'index.php';
        return;
    }
    
    selectedPackage = JSON.parse(savedPackage);
    
    // Araç yılı değiştiğinde marka listesini güncelle
    const yearSelect = document.getElementById('vehicleYear');
    if (yearSelect) {
        yearSelect.addEventListener('change', updateBrandList);
    }
    
    // Marka değiştiğinde model listesini güncelle
    const brandSelect = document.getElementById('vehicleBrand');
    if (brandSelect) {
        brandSelect.addEventListener('change', updateModelList);
    }
}

// Marka listesini güncelle
function updateBrandList() {
    const yearSelect = document.getElementById('vehicleYear');
    const brandSelect = document.getElementById('vehicleBrand');
    const modelSelect = document.getElementById('vehicleModel');
    
    const selectedYear = yearSelect.value;
    
    if (selectedYear) {
        // Bu kısım PHP'den AJAX ile marka listesi çekecek
        fetchBrands(selectedYear).then(brands => {
            brandSelect.innerHTML = '<option value="">Marka Seçin</option>';
            brands.forEach(brand => {
                brandSelect.innerHTML += `<option value="${brand.id}">${brand.name}</option>`;
            });
            brandSelect.disabled = false;
        });
    } else {
        brandSelect.innerHTML = '<option value="">Önce Yıl Seçin</option>';
        brandSelect.disabled = true;
    }
    
    // Model listesini sıfırla
    modelSelect.innerHTML = '<option value="">Önce Marka Seçin</option>';
    modelSelect.disabled = true;
}

// Model listesini güncelle
function updateModelList() {
    const brandSelect = document.getElementById('vehicleBrand');
    const modelSelect = document.getElementById('vehicleModel');
    
    const selectedBrand = brandSelect.value;
    
    if (selectedBrand) {
        // Bu kısım PHP'den AJAX ile model listesi çekecek
        fetchModels(selectedBrand).then(models => {
            modelSelect.innerHTML = '<option value="">Model Seçin</option>';
            models.forEach(model => {
                modelSelect.innerHTML += `<option value="${model.id}">${model.name}</option>`;
            });
            modelSelect.disabled = false;
        });
    } else {
        modelSelect.innerHTML = '<option value="">Önce Marka Seçin</option>';
        modelSelect.disabled = true;
    }
}

// AJAX fonksiyonları (PHP ile iletişim)
async function fetchBrands(year) {
    try {
        const response = await fetch(`ajax/get-brands.php?year=${year}`);
        return await response.json();
    } catch (error) {
        console.error('SMS gönderme hatası:', error);
        return { success: false, message: 'SMS gönderilemedi' };
    }
}

// Mesaj gösterme fonksiyonu
function showMessage(message, type = 'info') {
    // Mevcut mesajları temizle
    const existingMessages = document.querySelectorAll('.message');
    existingMessages.forEach(msg => msg.remove());
    
    const messageDiv = document.createElement('div');
    messageDiv.className = `message message-${type}`;
    messageDiv.innerHTML = `
        <span>${message}</span>
        <button onclick="this.parentElement.remove()" class="message-close">&times;</button>
    `;
    
    document.body.insertBefore(messageDiv, document.body.firstChild);
    
    // 5 saniye sonra otomatik kaldır
    setTimeout(() => {
        if (messageDiv.parentElement) {
            messageDiv.remove();
        }
    }, 5000);
}

// Loading gösterici
function showLoading(element) {
    element.disabled = true;
    element.innerHTML = `
        <span class="loading-spinner"></span>
        Yükleniyor...
    `;
}

function hideLoading(element, originalText) {
    element.disabled = false;
    element.innerHTML = originalText;
}

// Form gönderme fonksiyonu
async function submitForm(formId, endpoint) {
    const form = document.getElementById(formId);
    const submitBtn = form.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    
    try {
        showLoading(submitBtn);
        
        const formData = new FormData(form);
        const response = await fetch(endpoint, {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            if (result.redirect) {
                window.location.href = result.redirect;
            } else {
                showMessage(result.message, 'success');
            }
        } else {
            showMessage(result.message, 'error');
        }
    } catch (error) {
        showMessage('İşlem sırasında bir hata oluştu.', 'error');
    } finally {
        hideLoading(submitBtn, originalText);
    }
}

// Yardımcı fonksiyonlar
function formatPrice(price) {
    return new Intl.NumberFormat('tr-TR', {
        style: 'currency',
        currency: 'TRY'
    }).format(price);
}

function formatDate(date) {
    return new Date(date).toLocaleDateString('tr-TR');
}

function validateTCNo(tcno) {
    if (tcno.length !== 11) return false;
    
    let sum = 0;
    for (let i = 0; i < 10; i++) {
        sum += parseInt(tcno[i]);
    }
    
    return sum % 10 === parseInt(tcno[10]);
}

function validatePhone(phone) {
    const phoneRegex = /^(\+90|0)?[5][0-9]{2}[0-9]{3}[0-9]{4}$/;
    return phoneRegex.test(phone.replace(/\s/g, ''));
}

function validateEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

// Sayfa geçiş animasyonları
function fadeOut(element, callback) {
    element.style.opacity = '0';
    element.style.transform = 'translateY(-20px)';
    setTimeout(() => {
        element.style.display = 'none';
        if (callback) callback();
    }, 300);
}

function fadeIn(element) {
    element.style.display = 'block';
    element.style.opacity = '0';
    element.style.transform = 'translateY(20px)';
    
    setTimeout(() => {
        element.style.opacity = '1';
        element.style.transform = 'translateY(0)';
    }, 10);
}

// Mobil menü kontrolü
function toggleMobileMenu() {
    const mobileMenu = document.getElementById('mobileMenu');
    if (mobileMenu) {
        mobileMenu.classList.toggle('active');
    }
}

// Scroll to top
function scrollToTop() {
    window.scrollTo({
        top: 0,
        behavior: 'smooth'
    });
}

// Print fonksiyonu (poliçe yazdırma için)
function printPolicy(policyId) {
    const printWindow = window.open(`print-policy.php?id=${policyId}`, '_blank');
    printWindow.onload = function() {
        printWindow.print();
    };
}

// Export fonksiyonları
window.selectPackage = selectPackage;
window.goBack = goBack;
window.goToCustomerInfo = goToCustomerInfo;
window.showMessage = showMessage;
window.submitForm = submitForm;
window.toggleMobileMenu = toggleMobileMenu;
window.scrollToTop = scrollToTop;
window.printPolicy = printPolicy; catch (error) {
        console.error('Marka listesi alınamadı:', error);
        return [];
    }
}

async function fetchModels(brandId) {
    try {
        const response = await fetch(`ajax/get-models.php?brand=${brandId}`);
        return await response.json();
    } catch (error) {
        console.error('Model listesi alınamadı:', error);
        return [];
    }
}

// Form validasyonları
function setupFormValidations() {
    // Telefon numarası formatı
    const phoneInputs = document.querySelectorAll('input[type="tel"]');
    phoneInputs.forEach(input => {
        input.addEventListener('input', function() {
            let value = this.value.replace(/\D/g, '');
            if (value.length > 10) value = value.slice(0, 10);
            
            if (value.length >= 7) {
                value = value.replace(/(\d{3})(\d{3})(\d{4})/, '$1 $2 $3');
            } else if (value.length >= 4) {
                value = value.replace(/(\d{3})(\d{3})/, '$1 $2');
            }
            
            this.value = value;
        });
    });
    
    // TC Kimlik formatı
    const tcInputs = document.querySelectorAll('input[name="tc_no"]');
    tcInputs.forEach(input => {
        input.addEventListener('input', function() {
            this.value = this.value.replace(/\D/g, '').slice(0, 11);
        });
    });
    
    // Plaka formatı
    const plateInputs = document.querySelectorAll('input[name="plate"]');
    plateInputs.forEach(input => {
        input.addEventListener('input', function() {
            this.value = this.value.toUpperCase();
        });
    });
}

// Ödeme formu işlemleri
function initializePaymentForm() {
    // Kredi kartı numarası formatı
    const cardNumberInput = document.getElementById('cardNumber');
    if (cardNumberInput) {
        cardNumberInput.addEventListener('input', function() {
            let value = this.value.replace(/\D/g, '');
            value = value.replace(/(\d{4})(?=\d)/g, '$1 ');
            this.value = value;
        });
    }
    
    // CVV formatı
    const cvvInput = document.getElementById('cvv');
    if (cvvInput) {
        cvvInput.addEventListener('input', function() {
            this.value = this.value.replace(/\D/g, '').slice(0, 3);
        });
    }
    
    // Son kullanma tarihi formatı
    const expiryInput = document.getElementById('expiry');
    if (expiryInput) {
        expiryInput.addEventListener('input', function() {
            let value = this.value.replace(/\D/g, '');
            if (value.length >= 2) {
                value = value.slice(0, 2) + '/' + value.slice(2, 4);
            }
            this.value = value;
        });
    }
}

// Login formu işlemleri
function initializeLoginForm() {
    const loginForm = document.getElementById('loginForm');
    if (loginForm) {
        loginForm.addEventListener('submit', handleLogin);
    }
    
    const forgotPasswordLink = document.getElementById('forgotPassword');
    if (forgotPasswordLink) {
        forgotPasswordLink.addEventListener('click', showForgotPasswordForm);
    }
}

function handleLogin(e) {
    e.preventDefault();
    
    const phone = document.getElementById('phone').value;
    const password = document.getElementById('password').value;
    
    if (!phone || !password) {
        showMessage('Lütfen tüm alanları doldurun.', 'error');
        return;
    }
    
    // PHP'ye AJAX ile gönder
    loginUser(phone, password);
}

async function loginUser(phone, password) {
    try {
        const response = await fetch('ajax/login.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ phone, password })
        });
        
        const result = await response.json();
        
        if (result.success) {
            window.location.href = 'customer-panel.php';
        } else {
            showMessage(result.message, 'error');
        }
    } catch (error) {
        showMessage('Giriş yapılırken bir hata oluştu.', 'error');
    }
}

function showForgotPasswordForm() {
    // Şifremi unuttum formunu göster
    const forgotForm = document.getElementById('forgotPasswordForm');
    if (forgotForm) {
        forgotForm.style.display = 'block';
    }
}

// SMS gönderme fonksiyonu
async function sendSMS(phone, message) {
    try {
        const response = await fetch('ajax/send-sms.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ phone, message })
        });
        
        return await response.json();
    }